/*
 *  max17040_battery.c
 *  fuel-gauge systems for lithium-ion (Li+) batteries
 *
 *  Copyright (C) 2009 Samsung Electronics
 *  Minkyu Kang <mk7.kang@samsung.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/mutex.h>
#include <linux/err.h>
#include <linux/i2c.h>
#include <linux/delay.h>
#include <linux/power_supply.h>
#include <linux/max17040_battery.h>
#include <linux/slab.h>

#define MAX17040_VCELL_MSB	0x02
#define MAX17040_VCELL_LSB	0x03
#define MAX17040_SOC_MSB	0x04
#define MAX17040_SOC_LSB	0x05
#define MAX17040_MODE_MSB	0x06
#define MAX17040_MODE_LSB	0x07
#define MAX17040_VER_MSB	0x08
#define MAX17040_VER_LSB	0x09
#define MAX17040_RCOMP_MSB	0x0C
#define MAX17040_RCOMP_LSB	0x0D
#define MAX17040_CMD_MSB	0xFE
#define MAX17040_CMD_LSB	0xFF

#define MAX17040_DELAY			msecs_to_jiffies(3000)
#define MAX17040_BATTERY_FULL	98
#define	POWER_OFF_VOLTAGE		3412500
#define	POWER_MAX_VOLTAGE		4300000

//#define	MAX17040_DEBUG

struct max17040_chip {
	struct i2c_client		*client;
	struct delayed_work		work;
	struct power_supply		battery;
	struct max17040_platform_data	*pdata;

	/* State Of Connect */
	int online;
	/* battery voltage */
	int vcell;
	/* battery capacity */
	int soc;
	/* State Of Charge */
	int status;
};

static int max17040_get_property(struct power_supply *psy,
			    enum power_supply_property psp,
			    union power_supply_propval *val)
{
	struct max17040_chip *chip = container_of(psy,
				struct max17040_chip, battery);

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val->intval = chip->status;
		break;
	case POWER_SUPPLY_PROP_ONLINE:
		val->intval = chip->online;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		val->intval = chip->vcell;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = chip->soc;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int max17040_write_reg(struct i2c_client *client, u8 reg, u16 value)
{
	struct i2c_msg msg;
	unsigned char buf[3];
	int ret;

	buf[0] 	= reg;
	buf[1] 	= (value >> 8) & 0xFF;
	buf[2] 	= (value     ) & 0xFF;

	msg.addr	= client->addr;
	msg.flags 	= 0;
	msg.len 	= 3;
	msg.buf 	= (unsigned char *)&buf[0];
	
	if((ret = i2c_transfer(client->adapter, &msg, 1)) < 0)	{
		dev_err(&client->dev, "I2C write error: (%d) reg : 0x%02X value: 0x%04X\n", ret, reg, value);
		return ret;
	}

	return	ret;
}

static int max17040_read_reg(struct i2c_client *client, u8 reg)
{
	struct i2c_msg	msg[2];
	int 			ret;
	unsigned char	temp[2];

	msg[0].addr 	= client->addr;
	msg[0].flags 	= 0;
	msg[0].len 		= 1;
	msg[0].buf 		= (unsigned char *)&reg;

	msg[1].addr 	= client->addr;
	msg[1].flags    = I2C_M_RD;
	msg[1].len 		= 2;
	msg[1].buf 		= (unsigned char *)&temp[0];

	if ((ret = i2c_transfer(client->adapter, msg, 2)) != 2) {
		dev_err(&client->dev, "I2C read error: (%d) reg: 0x%02X\n", ret, reg);
		return -EIO;
	}

    ret = ((temp[0] << 8) & 0xFF00) | (temp[1] & 0xFF);

	return ret;
}

static void max17040_reset(struct i2c_client *client)
{
	max17040_write_reg(client, MAX17040_CMD_MSB,    0x5400);
}

static void max17040_get_vcell(struct i2c_client *client)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	chip->vcell = (max17040_read_reg(client, MAX17040_VCELL_MSB) >> 4) & 0xFFF;
	chip->vcell = (chip->vcell * 1250);

	#if defined(MAX17040_DEBUG)
		printk("chip->vcell = %d\n", chip->vcell);
	#endif
}

static void max17040_get_soc(struct i2c_client *client)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	chip->soc = (max17040_read_reg(client, MAX17040_SOC_MSB) >> 8) & 0xFF;

	#if defined(MAX17040_DEBUG)
		printk("chip->soc = %d\n", chip->soc);
	#endif

//	chip->soc = (chip->soc - 2) * 100 / (100 - 2);
	
	if(chip->soc > 100) chip->soc = 100;
	if(chip->soc < 0  ) chip->soc = 0;

	if(chip->soc)	{
		if(chip->vcell < POWER_OFF_VOLTAGE)		chip->soc = 0;
		if(chip->soc > MAX17040_BATTERY_FULL)	{
			if(chip->vcell < POWER_MAX_VOLTAGE)	chip->soc = MAX17040_BATTERY_FULL;
			else								chip->soc = MAX17040_BATTERY_FULL + 1;
		}
	}
	else	{
		if(chip->vcell >= POWER_OFF_VOLTAGE)	chip->soc = 1;
	}
}

static void max17040_get_version(struct i2c_client *client)
{
    int ver;
	
	ver = max17040_read_reg(client, MAX17040_VER_MSB);

	dev_info(&client->dev, "MAX17040 Fuel-Gauge Ver %04X\n", ver);

    // Hardkerenl Odroid-A4 RCOMP
    #if defined(CONFIG_BOARD_ODROID_A4)
        max17040_write_reg(client, MAX17040_RCOMP_MSB, 0xD000);
    	dev_info(&client->dev, "MAX17040 RCOMP Value %04X\n", max17040_read_reg(client, MAX17040_RCOMP_MSB));
    #endif
}

static void max17040_get_online(struct i2c_client *client)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	chip->online = 1;

	if(chip->pdata)	{
		if (chip->pdata->battery_online)
			chip->online = chip->pdata->battery_online();
	}
}

static void max17040_get_status(struct i2c_client *client)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	if (chip->soc > MAX17040_BATTERY_FULL)
		chip->status = POWER_SUPPLY_STATUS_FULL;

	if (chip->pdata)	{
		if (!chip->pdata->charger_online || !chip->pdata->charger_enable) {
			chip->status = POWER_SUPPLY_STATUS_UNKNOWN;
			return;
		}
	
		if (chip->pdata->charger_online()) {
			if (chip->pdata->charger_enable())	
				chip->status = POWER_SUPPLY_STATUS_CHARGING;
			else
				chip->status = POWER_SUPPLY_STATUS_NOT_CHARGING;
		} else {
			if(chip->status != POWER_SUPPLY_STATUS_DISCHARGING)
    			chip->status = POWER_SUPPLY_STATUS_DISCHARGING;
		}
	}
	else	{
		chip->status = POWER_SUPPLY_STATUS_DISCHARGING;
	}
}

static void max17040_work(struct work_struct *work)
{
	struct max17040_chip *chip = container_of(work, struct max17040_chip, work.work);

	int old_vcell = chip->vcell, old_soc = chip->soc;

	max17040_get_vcell(chip->client);
	max17040_get_soc(chip->client);
	max17040_get_online(chip->client);
	max17040_get_status(chip->client);

	if((old_vcell != chip->vcell) || (old_soc != chip->soc))	{
		power_supply_changed(&chip->battery);
		#if defined(MAX17040_DEBUG)
			printk("%s : power_supply_changed!\n", __func__);
		#endif		
	}
	schedule_delayed_work(&chip->work, MAX17040_DELAY);
}

static enum power_supply_property max17040_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_ONLINE,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
};

static int __devinit max17040_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct i2c_adapter *adapter = to_i2c_adapter(client->dev.parent);
	struct max17040_chip *chip;
	int ret;

	if (!i2c_check_functionality(adapter, I2C_FUNC_SMBUS_BYTE))
		return -EIO;

	chip = kzalloc(sizeof(*chip), GFP_KERNEL);
	if (!chip)
		return -ENOMEM;

	chip->client = client;
	chip->pdata = client->dev.platform_data;

	i2c_set_clientdata(client, chip);

	chip->battery.name				= "max17040-battery";
	chip->battery.type				= POWER_SUPPLY_TYPE_BATTERY;
	chip->battery.get_property		= max17040_get_property;
	chip->battery.properties		= max17040_battery_props;
	chip->battery.num_properties	= ARRAY_SIZE(max17040_battery_props);

	ret = power_supply_register(&client->dev, &chip->battery);
	if (ret) {
		dev_err(&client->dev, "failed: power supply register\n");
		kfree(chip);
		return ret;
	}
	else	{
		dev_info(&client->dev, "power supply max17040-battery registerd.\n");
	}

//	max17040_reset(client);
    
	max17040_get_version(client);

	INIT_DELAYED_WORK_DEFERRABLE(&chip->work, max17040_work);
	schedule_delayed_work(&chip->work, MAX17040_DELAY);

	return 0;
}

static int __devexit max17040_remove(struct i2c_client *client)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	power_supply_unregister(&chip->battery);
	cancel_delayed_work(&chip->work);
	kfree(chip);
	return 0;
}

#ifdef CONFIG_PM

static int max17040_suspend(struct i2c_client *client,
		pm_message_t state)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	cancel_delayed_work(&chip->work);
	return 0;
}

static int max17040_resume(struct i2c_client *client)
{
	struct max17040_chip *chip = i2c_get_clientdata(client);

	schedule_delayed_work(&chip->work, MAX17040_DELAY);
	return 0;
}

#else

#define max17040_suspend NULL
#define max17040_resume NULL

#endif /* CONFIG_PM */

static const struct i2c_device_id max17040_id[] = {
	{ "max17040", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, max17040_id);

static struct i2c_driver max17040_i2c_driver = {
	.driver	= {
		.name	= "max17040",
	},
	.probe		= max17040_probe,
	.remove		= __devexit_p(max17040_remove),
	.suspend	= max17040_suspend,
	.resume		= max17040_resume,
	.id_table	= max17040_id,
};

static int __init max17040_init(void)
{
	return i2c_add_driver(&max17040_i2c_driver);
}
module_init(max17040_init);

static void __exit max17040_exit(void)
{
	i2c_del_driver(&max17040_i2c_driver);
}
module_exit(max17040_exit);

MODULE_AUTHOR("Hardkernel Co,.Ltd");
MODULE_DESCRIPTION("MAX17040 Fuel Gauge");
MODULE_LICENSE("GPL");
