<?php
if (! class_exists('syntax_plugin_lang')) {
	if (! defined('DOKU_PLUGIN')) {
		if (! defined('DOKU_INC')) {
			define('DOKU_INC', realpath(dirname(__FILE__) . '/../../') . '/');
		} // if
		define('DOKU_PLUGIN', DOKU_INC . 'lib/plugins/');
	} // if
	// include parent class
	require_once(DOKU_PLUGIN . 'syntax.php');

/**
 * <tt>syntax_plugin_lang.php </tt>- A PHP4 class that implements
 * a <tt>DokuWiki</tt> plugin to specify an area using a different
 * language than the remaining document.
 *
 * <p>
 * Markup a section of text to be using a different language,
 * <tt>lang 2-letter-lang-code</tt>
 * </p><pre>
 *	Copyright (C) 2005, 2007 DFG/M.Watermann, D-10247 Berlin, FRG
 *			All rights reserved
 *		EMail : &lt;support@mwat.de&gt;
 * </pre>
 * <div class="disclaimer">
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either
 * <a href="http://www.gnu.org/licenses/gpl.html">version 3</a> of the
 * License, or (at your option) any later version.<br>
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * </div>
 * @author <a href="mailto:support@mwat.de">Matthias Watermann</a>
 * @version <tt>$Id: syntax_plugin_lang.php,v 1.4 2007/08/15 12:36:19 matthias Exp $</tt>
 * @since created 1-Sep-2005
 */
class syntax_plugin_lang extends DokuWiki_Syntax_Plugin {

	/**
	 * @publicsection
	 */
	//@{

	/**
	 * Tell the parser whether the plugin accepts syntax mode
	 * <tt>$aMode</tt> within its own markup.
	 *
	 * @param $aMode String The requested syntaxmode.
	 * @return Boolean <tt>TRUE</tt> unless <tt>$aMode</tt> is
	 * <tt>plugin_lang</tt> (which would result in a
	 * <tt>FALSE</tt> method result).
	 * @public
	 * @see getAllowedTypes()
	 * @static
	 */
	function accepts($aMode) {
		return ('plugin_lang' != $aMode);
	} // accepts()

	/**
	 * Connect lookup pattern to lexer.
	 *
	 * @param $aMode String The desired rendermode.
	 * @public
	 * @see render()
	 */
	function connectTo($aMode) {
		// See http://www.w3.org/TR/html401/struct/dirlang.html#h-8.1.1;
		// better (specialized) REs are used in 'handle()' method.
		$this->Lexer->addEntryPattern(
			'\x3Clang\s+[a-z\-A-Z0-9]{2,})?\s*\x3E\s*(?=(?s).*?\x3C\x2Flang\x3E)',
			$aMode, 'plugin_lang');
	} // connectTo()

	/**
	 * Get an associative array with plugin info.
	 *
	 * <p>
	 * The returned array holds the following fields:
	 * <dl>
	 * <dt>author</dt><dd>Author of the plugin</dd>
	 * <dt>email</dt><dd>Email address to contact the author</dd>
	 * <dt>date</dt><dd>Last modified date of the plugin in
	 * <tt>YYYY-MM-DD</tt> format</dd>
	 * <dt>name</dt><dd>Name of the plugin</dd>
	 * <dt>desc</dt><dd>Short description of the plugin (Text only)</dd>
	 * <dt>url</dt><dd>Website with more information on the plugin
	 * (eg. syntax description)</dd>
	 * </dl>
	 * @return Array Information about this plugin class.
	 * @public
	 * @static
	 */
	function getInfo() {
		return array(
			'author' =>	'Matthias Watermann',
			'email' =>	'support@mwat.de',
			'date' =>	'2007-08-15',
			'name' =>	'LANGuage Syntax Plugin',
			'desc' =>	'Markup a text area using another language',
			'url' =>	'http://wiki.splitbrain.org/plugin:lang');
	} // getInfo()

	/**
	 * Where to sort in?
	 *
	 * @return Integer <tt>498</tt> (doesn't really matter).
	 * @public
	 * @static
	 */
	function getSort() {
		return 498;
	} // getSort()

	/**
	 * Get the type of syntax this plugin defines.
	 *
	 * @return String <tt>'formatting'</tt>.
	 * @public
	 * @static
	 */
	function getType() {
		return 'formatting';
	} // getType()

	/**
	 * Handler to prepare matched data for the rendering process.
	 *
	 * <p>
	 * The <tt>$aState</tt> parameter gives the type of pattern
	 * which triggered the call to this method:
	 * </p>
	 * <dl>
	 * <dt>DOKU_LEXER_ENTER</dt>
	 * <dd>a pattern set by <tt>addEntryPattern()</tt></dd>
	 * <dt>DOKU_LEXER_MATCHED</dt>
	 * <dd>a pattern set by <tt>addPattern()</tt></dd>
	 * <dt>DOKU_LEXER_EXIT</dt>
	 * <dd> a pattern set by <tt>addExitPattern()</tt></dd>
	 * <dt>DOKU_LEXER_SPECIAL</dt>
	 * <dd>a pattern set by <tt>addSpecialPattern()</tt></dd>
	 * <dt>DOKU_LEXER_UNMATCHED</dt>
	 * <dd>ordinary text encountered within the plugin's syntax mode
	 * which doesn't match any pattern.</dd>
	 * </dl>
	 * @param $aMatch String The text matched by the patterns.
	 * @param $aState Integer The lexer state for the match.
	 * @param $aPos Integer The character position of the matched text.
	 * @param $aHandler Object Reference to the Doku_Handler object.
	 * @return Array Index <tt>[0]</tt> holds the current
	 * <tt>$aState</tt>, index <tt>[1]</tt> the match prepared for
	 * the <tt>render()</tt> method.
	 * @public
	 * @see render()
	 * @static
	 */
	function handle($aMatch, $aState, $aPos, &$aHandler) {
		if (DOKU_LEXER_ENTER == $aState) {
			$hits = array();
			// RFC 3066, "2. The Language tag", p. 2f.
			// Language-Tag = Primary-subtag *( "-" Subtag )
			if (preg_match('|\s+([a-z]{2,3})\s*>|i', $aMatch, $hits)) {
				// primary _only_ (most likely to be used)
				return array($aState, $hits[1]);
			} // if
			if (preg_match('|\s+([a-z]{2,3}\-[a-z0-9]{2,})\s*>|i',
			$aMatch, $hits)) {
				// primary _and_ subtag
				return array($aState, $hits[1]);
			} // if
			if (preg_match('|\s+([ix]\-[a-z0-9]{2,})\s*>|i', $aMatch, $hits)) {
				// 1-letter primary _and_ subtag
				return array($aState, $hits[1]);
			} // if
			if (preg_match('|\s+([a-z]{2,3})\-.*\s*>|i', $aMatch, $hits)) {
				// convenience: accept primary with empty subtag 
				return array($aState, $hits[1]);
			} // if
			// invalid language specification
			return array($aState, FALSE);
		} // if
		return array($aState, $aMatch);
	} // handle()

	/**
	 * Add exit pattern to lexer.
	 *
	 * @public
	 */
	function postConnect() {
		$this->Lexer->addExitPattern('\x3C\x2Flang\x3E', 'plugin_lang');
	} // postConnect()

	/**
	 * Handle the actual output creation.
	 *
	 * <p>
	 * The method checks for the given <tt>$aFormat</tt> and returns
	 * <tt>FALSE</tt> when a format isn't supported. <tt>$aRenderer</tt>
	 * contains a reference to the renderer object which is currently
	 * handling the rendering. The contents of <tt>$aData</tt> is the
	 * return value of the <tt>handle()</tt> method.
	 * </p>
	 * @param $aFormat String The output format to generate.
	 * @param $aRenderer Object A reference to the renderer object.
	 * @param $aData Array The data created by the <tt>handle()</tt>
	 * method.
	 * @return Boolean <tt>TRUE</tt> if rendered successfully, or
	 * <tt>FALSE</tt> otherwise.
	 * @public
	 * @see handle()
	 *
	 */
	function render($aFormat, &$aRenderer, &$aData) {
		if ('xhtml' != $aFormat) {
			return FALSE;
		} // if
		static $VALID = TRUE;	// flag to notice invalid markup
		switch ($aData[0]) {
			case DOKU_LEXER_ENTER:
				if ($aData[1]) {
					$aRenderer->doc .= '<span lang="' . $aData[1]
						. '" xml:lang="' . $aData[1] . '">';
				} else {
					$VALID = FALSE;
				} // if
				return TRUE;
			case DOKU_LEXER_UNMATCHED:
				$aRenderer->doc .= str_replace(array('&','<', '>'),
					array('&#38;', '&#60;', '&#62;'), $aData[1]);
				return TRUE;
			case DOKU_LEXER_EXIT:
				if ($VALID) {
					$aRenderer->doc .= '</span>';
				} else {
					$VALID = TRUE;
				} // if
			default:
				return TRUE;
		} // switch
	} // render()

	//@}
} // class syntax_plugin_lang
} // if
?>
